/* ------------------------------------------------------------------------------
  File: chr6dm_startup.c
  Author: CH Robotics
  Version: 1.0
  
  Description: Function definitions for AHRS initialization
------------------------------------------------------------------------------ */ 
#include "stm32f10x.h"
#include "UM6_startup.h"
#include "UM6_usart.h"
#include "CHR_i2c.h"

uint8_t gClockInUse;

/*******************************************************************************
* Function Name  : Initialize_IMU
* Input          : None
* Output         : None
* Return         : None
* Description    : Initializes the system clock and all peripherals of the CHR-6d
*******************************************************************************/
void Initialize_UM6( void )
{
	 ADC_InitTypeDef ADC_InitStructure;
    DMA_InitTypeDef DMA_InitStructure;
	 TIM_TimeBaseInitTypeDef  TIM_TimeBaseStructure;
	 I2C_InitTypeDef   I2C_InitStructure;
	 EXTI_InitTypeDef EXTI_InitStructure;
	 	 
	 // System Clocks Configuration
    RCC_Configuration();
       
    // NVIC configuration
    NVIC_Configuration();

    // Configure the GPIO ports 
    GPIO_Configuration();

	 // Setup serial port with 115200 baud	 
	 USART1_Configuration(115200);
	 
	 // i2c TX DMA initialization
	 DMA_DeInit(DMA1_Channel6);
	 DMA_InitStructure.DMA_PeripheralBaseAddr = (uint32_t)I2C1_DR_Address;
	 DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)g_i2cTxBuf;
	 DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralDST;
	 DMA_InitStructure.DMA_BufferSize = 0;		// This will need to be set on each transfer
	 DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	 DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
	 DMA_InitStructure.DMA_PeripheralDataSize = DMA_MemoryDataSize_Byte;
	 DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
	 DMA_InitStructure.DMA_Mode = DMA_Mode_Normal;
	 DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
	 DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
	 DMA_Init(DMA1_Channel6, &DMA_InitStructure);

	 // i2c RX DMA initialization
	 DMA_DeInit(DMA1_Channel7);
	 DMA_InitStructure.DMA_PeripheralBaseAddr = (uint32_t)I2C1_DR_Address;
	 DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)g_i2cRxBuf;
	 DMA_InitStructure.DMA_BufferSize = 0;		// This will need to be set on each transfer
	 DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	 DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
	 DMA_InitStructure.DMA_PeripheralDataSize = DMA_MemoryDataSize_Byte;
	 DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
	 DMA_InitStructure.DMA_Mode = DMA_Mode_Normal;
	 DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
	 DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
	 DMA_Init(DMA1_Channel7, &DMA_InitStructure);
	 
	 // Enable interrupts for i2c transfer complete
//	 DMA_ITConfig( DMA1_Channel6, DMA_IT_TC, ENABLE );
//	 DMA_ITConfig( DMA1_Channel7, DMA_IT_TC, ENABLE );
	 	 
	 // Configure Timer2 for state transmission
/*	 TIM_TimeBaseStructure.TIM_Period = 30000;
	 TIM_TimeBaseStructure.TIM_Prescaler = 100;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
	*/ 
	 TIM_TimeBaseStructure.TIM_Period = 1000;
	 TIM_TimeBaseStructure.TIM_Prescaler = 1;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;

	 TIM_TimeBaseInit(TIM2, &TIM_TimeBaseStructure);

	 // Configure Timer3 for prediction loop (used to compute elapsed time between
	 // state measurement updates)
	 TIM_TimeBaseStructure.TIM_Period = 0xFFFF;
	 TIM_TimeBaseStructure.TIM_Prescaler = 72;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;

	 TIM_TimeBaseInit(TIM3, &TIM_TimeBaseStructure);
	 
	 // Configure Timer4 for i2c timeout detection (timeout after 50 milliseconds)
	 TIM_TimeBaseStructure.TIM_Period = 50000;
	 TIM_TimeBaseStructure.TIM_Prescaler = 72;
	 TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	 TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;

	 TIM_TimeBaseInit(TIM4, &TIM_TimeBaseStructure);
	 TIM_ITConfig(TIM4, TIM_IT_Update, ENABLE);
	 
	 // Pull Self-Test pin low
	 GPIO_WriteBit( GPIOA, GPIO_Pin_13, Bit_RESET );
/*	 
	 // Enable I2C
	 I2C_Cmd(I2C1, ENABLE);
	 
	 // I2C1 configuration
	 I2C_InitStructure.I2C_Mode = I2C_Mode_I2C;
	 I2C_InitStructure.I2C_DutyCycle = I2C_DutyCycle_2;
	 I2C_InitStructure.I2C_OwnAddress1 = THIS_SLAVE_ADDRESS7;
	 I2C_InitStructure.I2C_Ack = I2C_Ack_Enable;
	 I2C_InitStructure.I2C_AcknowledgedAddress = I2C_AcknowledgedAddress_7bit;
	 I2C_InitStructure.I2C_ClockSpeed = I2C_CLOCK_SPEED;
	 I2C_Init(I2C1, &I2C_InitStructure);
	 */
	 
	 // Enable I2C1 event and buffer interrupts
	 // DISABLED FOR DMA USAGE
//	 I2C_ITConfig(I2C1, I2C_IT_EVT | I2C_IT_BUF, ENABLE);
	 
	 // Enable external interrupt 12 (DRDY pin for rate gyros)
	 // Map PA12 to EXTI 12
	 GPIO_EXTILineConfig(GPIO_PortSourceGPIOA, GPIO_PinSource12);

}

/*******************************************************************************
* Function Name  : USART_Init( int baud )
* Description    : Configures the USART at the selected baud rate
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void USART1_Configuration(int baud)
{
	 USART_InitTypeDef USART_InitStructure;
	 DMA_InitTypeDef DMA_InitStructure;
	 
	 // USART1 RX DMA1 Channel (triggered by USART1 Rx event) Config 
	 DMA_DeInit(DMA1_Channel5);
	 DMA_StructInit(&DMA_InitStructure);
	 DMA_InitStructure.DMA_PeripheralBaseAddr = USART1_DR_Base;
	 DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)gRXBuf;
	 DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralSRC;
	 DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
	 DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	 DMA_InitStructure.DMA_BufferSize = RX_BUF_SIZE;
	 DMA_InitStructure.DMA_Mode = DMA_Mode_Circular;
	 DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
	 DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
	 
	 DMA_Init( DMA1_Channel5, &DMA_InitStructure );
	 
	 // USART1 TX DMA1 channel config (triggered by USART1 Tx event)
	 DMA_DeInit(DMA1_Channel4);
	 DMA_InitStructure.DMA_PeripheralBaseAddr = USART1_DR_Base;
	 DMA_InitStructure.DMA_MemoryBaseAddr = (uint32_t)gTXBuf;
	 DMA_InitStructure.DMA_DIR = DMA_DIR_PeripheralDST;
	 DMA_InitStructure.DMA_BufferSize = TX_BUF_SIZE;		// This will need to be set again
	 DMA_InitStructure.DMA_PeripheralInc = DMA_PeripheralInc_Disable;
	 DMA_InitStructure.DMA_MemoryInc = DMA_MemoryInc_Enable;
	 DMA_InitStructure.DMA_PeripheralDataSize = DMA_PeripheralDataSize_Byte;
	 DMA_InitStructure.DMA_MemoryDataSize = DMA_MemoryDataSize_Byte;
	 DMA_InitStructure.DMA_Mode = DMA_Mode_Normal;
	 DMA_InitStructure.DMA_Priority = DMA_Priority_VeryHigh;
	 DMA_InitStructure.DMA_M2M = DMA_M2M_Disable;
	 DMA_Init(DMA1_Channel4, &DMA_InitStructure);
	 
	 // Enable TX DMA transfer complete interrupt
	 DMA_ITConfig( DMA1_Channel4, DMA_IT_TC, ENABLE );
	 
	 // Disable USART1
    USART_Cmd(USART1, DISABLE);
	 
	 // USART1  ------------------------------------------------------
    /* USART1 is configured as followS:
        - BaudRate = 'baud'  
        - Word Length = 8 Bits
        - One Stop Bit
        - No parity
        - Hardware flow control disabled (RTS and CTS signals)
        - Receive and transmit enabled
    */
    USART_InitStructure.USART_BaudRate = baud;
    USART_InitStructure.USART_WordLength = USART_WordLength_8b;
    USART_InitStructure.USART_StopBits = USART_StopBits_1;
    USART_InitStructure.USART_Parity = USART_Parity_No;
    USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
    USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;

    // Configure USART
    USART_DeInit(USART1);
	 USART_Init(USART1, &USART_InitStructure);
	 
    // Enable USART1 Receive and Transmit interrupts
//    USART_ITConfig(USART1, USART_IT_RXNE, ENABLE);
//    USART_ITConfig(USART1, USART_IT_TC, ENABLE);

	 // Enable USART1 TX and RX DMA requests
	 USART_DMACmd(USART1, USART_DMAReq_Rx, ENABLE);
	 USART_DMACmd(USART1, USART_DMAReq_Tx, ENABLE);
	 
	 // Enable DMA1 USART RX channel
	 DMA_Cmd(DMA1_Channel5, ENABLE);
	 
	 // Enable DMA1 USART TX channel (this should be enabled when there is data to transmit over the
	 // USART.  Transmission will start automatically.
//	 DMA_Cmd(DMA1_Channel4, ENABLE);

	 // Set global flags that indicate the USART status
	 gTXBusy = 0;
	 gRXBufPtr = 0;
	 gTXBufPtr = 0;
	 gUSART_State = USART_STATE_WAIT;
    
    // Enable USART1
    USART_Cmd(USART1, ENABLE);
}

/*******************************************************************************
* Function Name  : RCC_Configuration
* Description    : Configures the different system clocks.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RCC_Configuration(void)
{
	 ErrorStatus HSEStartUpStatus;
	 
	 RCC_DeInit();
	 
    /* Disable HSI */
//    RCC_HSICmd(ENABLE);
	 
	 /* Enable external oscillator */
	 RCC_HSEConfig(RCC_HSE_ON);
	 
	 /* Wait till HSE is ready */
	 HSEStartUpStatus = RCC_WaitForHSEStartUp();
//	 while( RCC_WaitForHSEStartUp() != SUCCESS );
	 
	 /* Enable Prefetch Buffer */
    FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable);

    /* Flash 2 wait state */
    FLASH_SetLatency(FLASH_Latency_2);

    /* HCLK = SYSCLK */
    RCC_HCLKConfig(RCC_SYSCLK_Div1);

    /* PCLK2 = HCLK/1 */
    RCC_PCLK2Config(RCC_HCLK_Div1); 

    /* PCLK1 = HCLK/2 */
    RCC_PCLK1Config(RCC_HCLK_Div2);

    /* ADCCLK = PCLK2/6 */
    RCC_ADCCLKConfig(RCC_PCLK2_Div6); 

	 if( HSEStartUpStatus == SUCCESS )
	 {
		  gClockInUse = 'E';
		  RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9);	// 8 / 1 * 9 = 72 Mhz
	 }
	 else
	 {
		  gClockInUse = 'I';
		  RCC_PLLConfig(RCC_PLLSource_HSI_Div2, RCC_PLLMul_16);
	 }
	 
    /* Enable PLL */ 
    RCC_PLLCmd(ENABLE);

    /* Wait till PLL is ready */
    while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET)
     {;}

    /* Select PLL as system clock source */
    RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);

    /* Wait till PLL is used as system clock source */
    while(RCC_GetSYSCLKSource() != 0x08)
     {;}

    /* Enable peripheral clocks --------------------------------------------------*/

    /* Enable GPIO clock */
    RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA | RCC_APB2Periph_GPIOB | RCC_APB2Periph_AFIO | RCC_APB2Periph_USART1, ENABLE);
    
    // Enable ADC1 and ADC2 clocks
    RCC_APB2PeriphClockCmd( RCC_APB2Periph_ADC1 | RCC_APB2Periph_ADC2, ENABLE );
                             
    // Enable DMA clock
    RCC_AHBPeriphClockCmd(RCC_AHBPeriph_DMA1, ENABLE);
			
	 // Enable TIM2 clock
	 RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM2, ENABLE);
			
	 // Enable TIM3 clock
	 RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM3, ENABLE);
			
	 // Enable TIM4 clock
	 RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM4, ENABLE);
			
	 // Enable i2c1 clock
	 RCC_APB1PeriphClockCmd(RCC_APB1Periph_I2C1, ENABLE);

}

/*******************************************************************************
* Function Name  : GPIO_Configuration
* Input          : None
* Output         : None
* Return         : None
* Description    : Configures the GPIO ports on the IMU
*******************************************************************************/
void GPIO_Configuration(void)
{
    GPIO_InitTypeDef GPIO_InitStructure;

    // Configure USART1 Rx as input floating
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
    GPIO_Init(GPIOA, &GPIO_InitStructure);

    // Configure USART1 Tx as alternate function push-pull
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_9;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // Configure USART2 Rx as input floating
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_3;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
    GPIO_Init(GPIOA, &GPIO_InitStructure);

    // Configure USART2 Tx as alternate function push-pull
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // LED output pin
	 GPIO_InitStructure.GPIO_Pin = GPIO_Pin_8 | GPIO_Pin_0;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // CONFIGURE A/D PINS.  A/D INPUTS COME IN THROUGH CHANNELS 1 THROUGH 9
/*
    // Configure PA.0,4,5,6,7 as analog inputs
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_4 | GPIO_Pin_5 | GPIO_Pin_6 | GPIO_Pin_7;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
    GPIO_Init(GPIOA, &GPIO_InitStructure);

    // Configure PB0 and PB1 as analog inputs
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
    GPIO_Init(GPIOB, &GPIO_InitStructure);
	 */ 	 
	 
	 
	 // Configure PA13 to use alternate function (default is
	 // JTAG - we want GPIO).  This is the interrupt line for the rate gyros.
	 GPIO_PinRemapConfig( GPIO_Remap_SWJ_JTAGDisable, ENABLE);
	 GPIO_PinRemapConfig( GPIO_Remap_SWJ_Disable , ENABLE);
	 GPIO_InitStructure.GPIO_Pin = GPIO_Pin_13;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // Configure PA12 as input floating (accel interrupt input)
	 GPIO_InitStructure.GPIO_Pin = GPIO_Pin_12;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	 GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // Configure PA0 as output PP
	 GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	 GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
    GPIO_Init(GPIOA, &GPIO_InitStructure);
	 
	 // Configure PA11 as input floating (magnetometer interrupt input)
	 GPIO_InitStructure.GPIO_Pin = GPIO_Pin_11;
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	 GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
    GPIO_Init(GPIOA, &GPIO_InitStructure);	 
	 
	 /* DISABLED - WE ARE GOING TO BIT-BANG THE I2C BUS
	 // Configure I2C1 pins: SCL and SDA
	 GPIO_InitStructure.GPIO_Pin =  GPIO_Pin_6 | GPIO_Pin_7;
	 GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	 GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_OD;
	 GPIO_Init(GPIOB, &GPIO_InitStructure);
	 */
	 
	 // Configure I2C1 pins: SCL and SDA
	 GPIO_InitStructure.GPIO_Pin =  GPIO_Pin_6 | GPIO_Pin_7;
	 GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	 GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_OD;
	 GPIO_Init(GPIOB, &GPIO_InitStructure);
}   


/*******************************************************************************
* Function Name  : NVIC_Configuration
* Input          : None
* Output         : None
* Return         : None
* Description    : Configures the vectored interrupt table on the IMU
*******************************************************************************/
void NVIC_Configuration(void)
{
    NVIC_InitTypeDef NVIC_InitStructure;

    /* Configure the NVIC Preemption Priority Bits */  
    NVIC_PriorityGroupConfig(NVIC_PriorityGroup_0);

	 /*	DISABLED - DMA IS USED INSTEAD
    // Enable the USART1 Interrupt (only peripheral with higher priority is the i2c bus)
    NVIC_InitStructure.NVIC_IRQChannel = USART1_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
	 */

    // Enable the DMA1 channel4 interrupt
    NVIC_InitStructure.NVIC_IRQChannel = DMA1_Channel4_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 2;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
	 
	 /*
	 // Enable the DMA1 channel6 interrupt
    NVIC_InitStructure.NVIC_IRQChannel = DMA1_Channel6_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
	 
	 // Enable the DMA1 channel7 interrupt
    NVIC_InitStructure.NVIC_IRQChannel = DMA1_Channel7_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
    NVIC_Init(&NVIC_InitStructure);
	 */
	 
	 // Enable the TIM2 global Interrupt and set at lowest priority.
	 // This is used to tell the MCU to transmit newest state data over the UART
	 NVIC_InitStructure.NVIC_IRQChannel = TIM2_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0x0F;
	 NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0x0F;
	 NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	 NVIC_Init(&NVIC_InitStructure);
	 
	 // Enable the TIM4 global Interrupt
	 // This is used to indicate that the i2c bus timed out
	 NVIC_InitStructure.NVIC_IRQChannel = TIM4_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0x00;
	 NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0x01;
	 NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	 NVIC_Init(&NVIC_InitStructure);
	 
	 /* DISABLED TO FOR DMA USAGE
	 // Configure and enable I2C1 event interrupt (don't let inaything interfere with i2c communication)
	 NVIC_InitStructure.NVIC_IRQChannel = I2C1_EV_IRQn;
	 NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
	 NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
	 NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	 NVIC_Init(&NVIC_InitStructure);
	 */
  
	 // Enable and set EXTI Interrupt for handling data_ready indications from sensors
    NVIC_InitStructure.NVIC_IRQChannel = EXTI15_10_IRQn;
    NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 1;
    NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
    NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;

    NVIC_Init(&NVIC_InitStructure);
}
